from django.contrib import admin
from .models import TransactionRequest
from .tasks import submit_to_goexch9  # we’ll create later
from django.utils.html import format_html
from django.urls import path
from django.shortcuts import redirect
from django.contrib import messages

# @admin.register(TransactionRequest)
# class DepositRequestAdmin(admin.ModelAdmin):
#     list_display = ("user_id", "amount", "status", "created_at")
#     actions = ["approve_request", "reject_request"]

#     def approve_request(self, request, queryset):
#         for deposit in queryset:
#             deposit.status = "approved"
#             deposit.save()
#             submit_to_goexch9(deposit)  # Trigger auto-submit
#     approve_request.short_description = "Approve selected deposits"

#     def reject_request(self, request, queryset):
#         queryset.update(status="rejected")
#     reject_request.short_description = "Reject selected deposits"
@admin.register(TransactionRequest)
class TransactionRequestAdmin(admin.ModelAdmin):
    list_display = ("user_id", "amount", "status", "txn_type", "created_at", "action_buttons")
    list_filter = ("status", "txn_type")
    readonly_fields = ("created_at", "updated_at")

    # Custom buttons in admin list
    def action_buttons(self, obj):
        if obj.status == "pending":
            return format_html(
                '<a class="button" href="{}">Approve</a>&nbsp;'
                '<a class="button" href="{}">Reject</a>',
                f'/admin/admin_panel_app/transactionrequest/{obj.id}/approve/',
                f'/admin/admin_panel_app/transactionrequest/{obj.id}/reject/'
            )
        return "Already processed"
    action_buttons.short_description = "Actions"
    action_buttons.allow_tags = True

    def get_urls(self):
        urls = super().get_urls()
        custom_urls = [
            path('<int:pk>/approve/', self.admin_site.admin_view(self.approve_transaction), name='transaction-approve'),
            path('<int:pk>/reject/', self.admin_site.admin_view(self.reject_transaction), name='transaction-reject'),
        ]
        return custom_urls + urls

    def approve_transaction(self, request, pk):
        txn = self.get_object(request, pk)
        if txn.status != "pending":
            self.message_user(request, "Transaction already processed!", level=messages.WARNING)
            return redirect(f'/admin/{txn._meta.app_label}/{txn._meta.model_name}/')
        
        # Update status
        txn.status = "approved"
        txn.save()

        # Optional: Run deposit in background thread
        from threading import Thread
        Thread(target=submit_to_goexch9, args=(txn.user_id, txn.amount)).start()

        self.message_user(request, f"Transaction {txn.id} approved and deposit started.", level=messages.SUCCESS)
        return redirect(f'/admin/{txn._meta.app_label}/{txn._meta.model_name}/')

    def reject_transaction(self, request, pk):
        txn = self.get_object(request, pk)
        if txn.status != "pending":
            self.message_user(request, "Transaction already processed!", level=messages.WARNING)
            return redirect(f'/admin/{txn._meta.app_label}/{txn._meta.model_name}/')

        txn.status = "rejected"
        txn.save()
        self.message_user(request, f"Transaction {txn.id} rejected.", level=messages.ERROR)
        return redirect(f'/admin/{txn._meta.app_label}/{txn._meta.model_name}/')